// @HEADER
// ************************************************************************
//
//                           Intrepid2 Package
//                 Copyright (2007) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Kyungjoo Kim  (kyukim@sandia.gov), or
//                    Mauro Perego  (mperego@sandia.gov)
//
// ************************************************************************
// @HEADER

/** \file   Intrepid2_CubatureDirectTetDefaultDef.hpp
    \brief  Definition file for the Intrepid2::CubatureDirectTetDefault class.
    \author Created by P. Bochev, D. Ridzal, and D. Day.
            Kokkorized by Kyungjoo Kim
*/

namespace Intrepid2 {

  template <typename DT, typename PT, typename WT>
  CubatureDirectTetDefault<DT,PT,WT>::
  CubatureDirectTetDefault(const ordinal_type degree)
    : CubatureDirect<DT>(degree, 3) {

    INTREPID2_TEST_FOR_EXCEPTION( degree < 0 ||
                                  degree > static_cast<ordinal_type>(Parameters::MaxCubatureDegreeTet), std::out_of_range,
                                  ">>> ERROR (CubatureDirectTetDefault): No cubature rule implemented for the desired polynomial degree.");

    typedef Kokkos::DynRankView<PT,Kokkos::LayoutRight,Kokkos::HostSpace> pointViewHostType;
    typedef Kokkos::DynRankView<WT,Kokkos::LayoutRight,Kokkos::HostSpace> weightViewHostType;

    this->cubatureData_.numPoints_ = cubatureDataStatic_[this->degree_].numPoints_;
    const Kokkos::pair<ordinal_type,ordinal_type> pointRange(0, this->cubatureData_.numPoints_);
    {
      // src
      const pointViewHostType points_host(const_cast<PT*>( &(cubatureDataStatic_[this->degree_].points_[0][0]) ),
                                          pointRange.second,
                                          Parameters::MaxDimension);

      auto points = Kokkos::create_mirror_view(typename DT::memory_space(), points_host);

      Kokkos::deep_copy(points,points_host);

      // dst
      this->cubatureData_.points_ = Kokkos::DynRankView<PT,DT>("CubatureDirectTetDefault::cubatureData_::points_",
                                                                pointRange.second,
                                                                Parameters::MaxDimension);
      // copy
      Kokkos::deep_copy(this->cubatureData_.points_ , points );
    }
    {
      // src
      const weightViewHostType weights(const_cast<PT*>( &(cubatureDataStatic_[this->degree_].weights_[0]) ),
                                       pointRange.second);

      // dst
      this->cubatureData_.weights_ = Kokkos::DynRankView<WT,DT>("CubatureDirectTetDefault::cubatureData_::weights_",
                                                                 pointRange.second);
      // copy
      Kokkos::deep_copy(Kokkos::subdynrankview(this->cubatureData_.weights_, Kokkos::ALL()) , Kokkos::subdynrankview(weights, Kokkos::ALL()));
    }
  }

  //-------------------------------------------------------------------------------------//
  //                          Definition of cubature templates                           //
  //-------------------------------------------------------------------------------------//
  
  /*
    Cubature templates for lines are defined the reference Tetrahedron:
    
    Tetrahedron    -> (0,0,0), (1,0,0), (0,1,0), (0,0,1)
  */
  
  /*
    This static const member contains templates for default tetrahedron rules.
  */
  
  template<typename DT, typename PT, typename WT>
  const typename CubatureDirect<DT,PT,WT>::CubatureDataStatic
  CubatureDirectTetDefault<DT,PT,WT>::
  cubatureDataStatic_[cubatureDataStaticSize] = {
    // Cubature templates for the reference tet {(0,0,0), (1,0,0), (0,1,0), (0,0,1)}
    //
    {
      1,
      {{1./4., 1./4., 1./4.}},
      {1./6.}
    },
    {
      1,
      {{1./4., 1./4., 1./4.}},
      {1./6.}
    },
    {
      4,
      {{0.1381966011250105151795413165634361882280, 0.1381966011250105151795413165634361882280, 0.1381966011250105151795413165634361882280},
       {0.5854101966249684544613760503096914353161, 0.1381966011250105151795413165634361882280, 0.1381966011250105151795413165634361882280},
       {0.1381966011250105151795413165634361882280, 0.5854101966249684544613760503096914353161, 0.1381966011250105151795413165634361882280},
       {0.1381966011250105151795413165634361882280, 0.1381966011250105151795413165634361882280, 0.5854101966249684544613760503096914353161}},
      {1./24.,
       1./24.,
       1./24.,
       1./24.}
    },
    {
      5,
      {{1./4., 1./4., 1./4.},
       {1./6., 1./6., 1./6.},
       {1./6., 1./6., 1./2.},
       {1./6., 1./2., 1./6.},
       {1./2., 1./6., 1./6.}},
      {-2./15.,
       3./40.,
       3./40.,
       3./40.,
       3./40.}
    },
    {
      11,
      {{2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {3.99403576166799e-01, 3.99403576166799e-01, 1.00596423833201e-01},
       {3.99403576166799e-01, 1.00596423833201e-01, 3.99403576166799e-01},
       {1.00596423833201e-01, 3.99403576166799e-01, 3.99403576166799e-01},
       {3.99403576166799e-01, 1.00596423833201e-01, 1.00596423833201e-01},
       {1.00596423833201e-01, 3.99403576166799e-01, 1.00596423833201e-01},
       {1.00596423833201e-01, 1.00596423833201e-01, 3.99403576166799e-01}},
      {-1.31555555555555e-02,
       7.62222222222225e-03,
       7.62222222222225e-03,
       7.62222222222225e-03,
       7.62222222222225e-03,
       2.48888888888889e-02,
       2.48888888888889e-02,
       2.48888888888889e-02,
       2.48888888888889e-02,
       2.48888888888889e-02,
       2.48888888888889e-02}
    },
    {
      14,
      {{9.2735250310891226402323913737030605245203e-2,  9.2735250310891226402323913737030605245203e-2, 9.2735250310891226402323913737030605245203e-2},
       {7.2179424906732632079302825878890818426439e-1,  9.2735250310891226402323913737030605245203e-2, 9.2735250310891226402323913737030605245203e-2},
       {9.2735250310891226402323913737030605245203e-2,  7.2179424906732632079302825878890818426439e-1, 9.2735250310891226402323913737030605245203e-2},
       {9.2735250310891226402323913737030605245203e-2,  9.2735250310891226402323913737030605245203e-2, 7.2179424906732632079302825878890818426439e-1},
       {3.1088591926330060979734573376345783299262e-1,  3.1088591926330060979734573376345783299262e-1, 3.1088591926330060979734573376345783299262e-1},
       {6.7342242210098170607962798709626501022148e-2,  3.1088591926330060979734573376345783299262e-1, 3.1088591926330060979734573376345783299262e-1},
       {3.1088591926330060979734573376345783299262e-1,  6.7342242210098170607962798709626501022148e-2, 3.1088591926330060979734573376345783299262e-1},
       {3.1088591926330060979734573376345783299262e-1,  3.1088591926330060979734573376345783299262e-1, 6.7342242210098170607962798709626501022148e-2},
       {4.5503704125649637643858844819262773124457e-2,  4.5503704125649637643858844819262773124457e-2, 4.5449629587435036235614115518073722687554e-1},
       {4.5503704125649689297853203692977039634167e-2,  4.5449629587435031070214679630702296036583e-1, 4.5503704125649637643858844819262773124457e-2},
       {4.5449629587435031070214679630702296036583e-1,  4.5503704125649689297853203692977039634167e-2, 4.5503704125649637643858844819262773124457e-2},
       {4.5503704125649689297853203692977039634167e-2,  4.5449629587435031070214679630702296036583e-1, 4.5449629587435036235614115518073722687554e-1},
       {4.5449629587435031070214679630702296036583e-1,  4.5503704125649689297853203692977039634167e-2, 4.5449629587435036235614115518073722687554e-1},
       {4.5449629587435036235614115518073722687554e-1,  4.5449629587435036235614115518073722687554e-1, 4.5503704125649637643858844819262773124457e-2}},
      {1.2248840519393658257285034247721250587182e-2,
       1.2248840519393658257285034247721250587182e-2,
       1.2248840519393658257285034247721250587182e-2,
       1.2248840519393658257285034247721250587182e-2,
       1.8781320953002641799864275388881055635078e-2,
       1.8781320953002641799864275388881055635078e-2,
       1.8781320953002641799864275388881055635078e-2,
       1.8781320953002641799864275388881055635078e-2,
       7.0910034628469110730115713533762402962712e-3,
       7.0910034628469110730115713533762402962712e-3,
       7.0910034628469110730115713533762402962712e-3,
       7.0910034628469110730115713533762402962712e-3,
       7.0910034628469110730115713533762402962712e-3,
       7.0910034628469110730115713533762402962712e-3}
    },
    {
      24,
      {{2.1460287125915202404641618104947039637455e-1,  2.1460287125915202404641618104947039637455e-1, 2.1460287125915202404641618104947039637455e-1},
       {3.5619138622254392786075145685158881087635e-1,  2.1460287125915202404641618104947039637455e-1, 2.1460287125915202404641618104947039637455e-1},
       {2.1460287125915202404641618104947039637455e-1,  3.5619138622254392786075145685158881087635e-1, 2.1460287125915202404641618104947039637455e-1},
       {2.1460287125915202404641618104947039637455e-1,  2.1460287125915202404641618104947039637455e-1, 3.5619138622254392786075145685158881087635e-1},
       {3.2233789014227551528749363980198720305472e-1,  3.2233789014227551528749363980198720305472e-1, 3.2233789014227551528749363980198720305472e-1},
       {3.2986329573173454137519080594038390835837e-2,  3.2233789014227551528749363980198720305472e-1, 3.2233789014227551528749363980198720305472e-1},
       {3.2233789014227551528749363980198720305472e-1,  3.2986329573173454137519080594038390835837e-2, 3.2233789014227551528749363980198720305472e-1},
       {3.2233789014227551528749363980198720305472e-1,  3.2233789014227551528749363980198720305472e-1, 3.2986329573173454137519080594038390835837e-2},
       {4.0673958534611354036686370754218389288312e-2,  4.0673958534611354036686370754218389288312e-2, 4.0673958534611354036686370754218389288312e-2},
       {8.7797812439616593788994088773734483213506e-1,  4.0673958534611354036686370754218389288312e-2, 4.0673958534611354036686370754218389288312e-2},
       {4.0673958534611354036686370754218389288312e-2,  8.7797812439616593788994088773734483213506e-1, 4.0673958534611354036686370754218389288312e-2},
       {4.0673958534611354036686370754218389288312e-2,  4.0673958534611354036686370754218389288312e-2, 8.7797812439616593788994088773734483213506e-1},
       {6.3661001875017529993415628745574107721625e-2,  6.3661001875017529993415628745574107721625e-2, 2.6967233145831579864573760344791209827003e-1},
       {6.3661001875017541929214705659242736905843e-2,  2.6967233145831578670993852653424346908581e-1, 6.3661001875017529993415628745574107721625e-2},
       {2.6967233145831578670993852653424346908581e-1,  6.3661001875017541929214705659242736905843e-2, 6.3661001875017529993415628745574107721625e-2},
       {6.3661001875017533122869029505472299577799e-2,  6.0300566479164914449688453982083787814289e-1, 2.6967233145831580177519100420781029012621e-1},
       {6.0300566479164914449688453982083787814289e-1,  6.3661001875017533122869029505472299577799e-2, 2.6967233145831580177519100420781029012621e-1},
       {6.3661001875017545058668106419140928762018e-2,  2.6967233145831578983939192729414166094199e-1, 6.0300566479164914449688453982083787814289e-1},
       {2.6967233145831578983939192729414166094199e-1,  6.3661001875017545058668106419140928762018e-2, 6.0300566479164914449688453982083787814289e-1},
       {6.0300566479164914449688453982083787814289e-1,  2.6967233145831580177519100420781029012621e-1, 6.3661001875017533122869029505472299577799e-2},
       {2.6967233145831580177519100420781029012621e-1,  6.0300566479164914449688453982083787814289e-1, 6.3661001875017533122869029505472299577799e-2},
       {6.0300566479164913510852433754114330257437e-1,  6.3661001875017523734508827225777724009276e-2, 6.3661001875017523734508827225777724009276e-2},
       {6.3661001875017523734508827225777724009276e-2,  6.0300566479164913510852433754114330257437e-1, 6.3661001875017523734508827225777724009276e-2},
       {6.3661001875017523734508827225777724009276e-2,  6.3661001875017523734508827225777724009276e-2, 6.0300566479164913510852433754114330257437e-1}},
      {6.6537917096945836907319255002212225177840e-3,
       6.6537917096945836907319255002212225177840e-3,
       6.6537917096945836907319255002212225177840e-3,
       6.6537917096945836907319255002212225177840e-3,
       9.2261969239424523160433682126100883383192e-3,
       9.2261969239424523160433682126100883383192e-3,
       9.2261969239424523160433682126100883383192e-3,
       9.2261969239424523160433682126100883383192e-3,
       1.6795351758867738832382653285812235376638e-3,
       1.6795351758867738832382653285812235376638e-3,
       1.6795351758867738832382653285812235376638e-3,
       1.6795351758867738832382653285812235376638e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3,
       8.0357142857142855922177025417513774242999e-3}
    },
    {
      31,
      {{5.00000000000000e-01, 5.00000000000000e-01, 0.00000000000000e+00},
       {5.00000000000000e-01, 0.00000000000000e+00, 5.00000000000000e-01},
       {0.00000000000000e+00, 5.00000000000000e-01, 5.00000000000000e-01},
       {0.00000000000000e+00, 0.00000000000000e+00, 5.00000000000000e-01},
       {0.00000000000000e+00, 5.00000000000000e-01, 0.00000000000000e+00},
       {5.00000000000000e-01, 0.00000000000000e+00, 0.00000000000000e+00},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {7.82131923303180e-02, 7.82131923303180e-02, 7.82131923303180e-02},
       {7.82131923303180e-02, 7.82131923303180e-02, 7.65360423009046e-01},
       {7.82131923303180e-02, 7.65360423009046e-01, 7.82131923303180e-02},
       {7.65360423009046e-01, 7.82131923303180e-02, 7.82131923303180e-02},
       {1.21843216663905e-01, 1.21843216663905e-01, 1.21843216663905e-01},
       {1.21843216663905e-01, 1.21843216663905e-01, 6.34470350008284e-01},
       {1.21843216663905e-01, 6.34470350008284e-01, 1.21843216663905e-01},
       {6.34470350008284e-01, 1.21843216663905e-01, 1.21843216663905e-01},
       {3.32539164446421e-01, 3.32539164446421e-01, 3.32539164446421e-01},
       {3.32539164446421e-01, 3.32539164446421e-01, 2.38250666073803e-03},
       {3.32539164446421e-01, 2.38250666073803e-03, 3.32539164446421e-01},
       {2.38250666073803e-03, 3.32539164446421e-01, 3.32539164446421e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 2.00000000000000e-01},
       {1.00000000000000e-01, 2.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 6.00000000000000e-01},
       {1.00000000000000e-01, 6.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 2.00000000000000e-01, 6.00000000000000e-01},
       {1.00000000000000e-01, 6.00000000000000e-01, 2.00000000000000e-01},
       {2.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {2.00000000000000e-01, 1.00000000000000e-01, 6.00000000000000e-01},
       {2.00000000000000e-01, 6.00000000000000e-01, 1.00000000000000e-01},
       {6.00000000000000e-01, 1.00000000000000e-01, 2.00000000000000e-01},
       {6.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {6.00000000000000e-01, 2.00000000000000e-01, 1.00000000000000e-01}},
      {9.70017636684250e-04,
       9.70017636684250e-04,
       9.70017636684250e-04,
       9.70017636684250e-04,
       9.70017636684250e-04,
       9.70017636684250e-04,
       1.82642234661089e-02,
       1.05999415244136e-02,
       1.05999415244136e-02,
       1.05999415244136e-02,
       1.05999415244136e-02,
       -6.25177401143319e-02,
       -6.25177401143319e-02,
       -6.25177401143319e-02,
       -6.25177401143319e-02,
       4.89142526307350e-03,
       4.89142526307350e-03,
       4.89142526307350e-03,
       4.89142526307350e-03,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02,
       2.75573192239859e-02}
    },
    {
      43,
      {{2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.06829931610673e-01, 2.06829931610673e-01, 2.06829931610673e-01},
       {2.06829931610673e-01, 2.06829931610673e-01, 3.79510205167981e-01},
       {2.06829931610673e-01, 3.79510205167981e-01, 2.06829931610673e-01},
       {3.79510205167981e-01, 2.06829931610673e-01, 2.06829931610673e-01},
       {8.21035883105465e-02, 8.21035883105465e-02, 8.21035883105465e-02},
       {8.21035883105465e-02, 8.21035883105465e-02, 7.53689235068360e-01},
       {8.21035883105465e-02, 7.53689235068360e-01, 8.21035883105465e-02},
       {7.53689235068360e-01, 8.21035883105465e-02, 8.21035883105465e-02},
       {5.78195050519797e-03, 5.78195050519797e-03, 5.78195050519797e-03},
       {5.78195050519797e-03, 5.78195050519797e-03, 9.82654148484406e-01},
       {5.78195050519797e-03, 9.82654148484406e-01, 5.78195050519797e-03},
       {9.82654148484406e-01, 5.78195050519797e-03, 5.78195050519797e-03},
       {5.05327400188940e-02, 5.05327400188940e-02, 4.49467259981106e-01},
       {5.05327400188940e-02, 4.49467259981106e-01, 5.05327400188940e-02},
       {4.49467259981106e-01, 5.05327400188940e-02, 5.05327400188940e-02},
       {5.05327400188940e-02, 4.49467259981106e-01, 4.49467259981106e-01},
       {4.49467259981106e-01, 5.05327400188940e-02, 4.49467259981106e-01},
       {4.49467259981106e-01, 4.49467259981106e-01, 5.05327400188940e-02},
       {2.29066536116811e-01, 2.29066536116811e-01, 3.56395827885340e-02},
       {2.29066536116811e-01, 3.56395827885340e-02, 2.29066536116811e-01},
       {2.29066536116811e-01, 2.29066536116811e-01, 5.06227344977843e-01},
       {2.29066536116811e-01, 5.06227344977843e-01, 2.29066536116811e-01},
       {2.29066536116811e-01, 3.56395827885340e-02, 5.06227344977843e-01},
       {2.29066536116811e-01, 5.06227344977843e-01, 3.56395827885340e-02},
       {3.56395827885340e-02, 2.29066536116811e-01, 2.29066536116811e-01},
       {3.56395827885340e-02, 2.29066536116811e-01, 5.06227344977843e-01},
       {3.56395827885340e-02, 5.06227344977843e-01, 2.29066536116811e-01},
       {5.06227344977843e-01, 2.29066536116811e-01, 3.56395827885340e-02},
       {5.06227344977843e-01, 2.29066536116811e-01, 2.29066536116811e-01},
       {5.06227344977843e-01, 3.56395827885340e-02, 2.29066536116811e-01},
       {3.66077495531975e-02, 3.66077495531975e-02, 1.90486041934633e-01},
       {3.66077495531975e-02, 1.90486041934633e-01, 3.66077495531975e-02},
       {3.66077495531975e-02, 3.66077495531975e-02, 7.36298458958972e-01},
       {3.66077495531975e-02, 7.36298458958972e-01, 3.66077495531975e-02},
       {3.66077495531975e-02, 1.90486041934633e-01, 7.36298458958972e-01},
       {3.66077495531975e-02, 7.36298458958972e-01, 1.90486041934633e-01},
       {1.90486041934633e-01, 3.66077495531975e-02, 3.66077495531975e-02},
       {1.90486041934633e-01, 3.66077495531975e-02, 7.36298458958972e-01},
       {1.90486041934633e-01, 7.36298458958972e-01, 3.66077495531975e-02},
       {7.36298458958972e-01, 3.66077495531975e-02, 1.90486041934633e-01},
       {7.36298458958972e-01, 3.66077495531975e-02, 3.66077495531975e-02},
       {7.36298458958972e-01, 1.90486041934633e-01, 3.66077495531975e-02}},
      {-2.05001886586399e-02,
       1.42503058228669e-02,
       1.42503058228669e-02,
       1.42503058228669e-02,
       1.42503058228669e-02,
       1.96703331313388e-03,
       1.96703331313388e-03,
       1.96703331313388e-03,
       1.96703331313388e-03,
       1.69834109092875e-04,
       1.69834109092875e-04,
       1.69834109092875e-04,
       1.69834109092875e-04,
       4.57968382446725e-03,
       4.57968382446725e-03,
       4.57968382446725e-03,
       4.57968382446725e-03,
       4.57968382446725e-03,
       4.57968382446725e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       5.70448580868188e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03,
       2.14051914116212e-03}
    },
    {
      126,
      {{7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       4.06808035714288e-03,
       4.06808035714288e-03,
       4.06808035714288e-03,
       4.06808035714288e-03,
       -9.40623162845000e-05}
    },
    {
      126,
      {{7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       4.06808035714288e-03,
       4.06808035714288e-03,
       4.06808035714288e-03,
       4.06808035714288e-03,
       -9.40623162845000e-05}
    },
    {
      126,
      {{7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       4.53628240650810e-02,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       -1.16523476523476e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       1.01937289979825e-01,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       -3.50253861364972e-02,
       4.06808035714288e-03,
       4.06808035714288e-03,
       4.06808035714288e-03,
       4.06808035714288e-03,
       -9.40623162845000e-05}
    },
    {
      210,
      {{6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       -6.65685876623375e-04,
       -6.65685876623375e-04,
       -6.65685876623375e-04,
       -6.65685876623375e-04,
       6.27082108562500e-06}
    },
    {
      210,
      {{6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       5.25509090165175e-02,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       -1.48185225279265e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       1.49815898387327e-01,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       -6.53444166537338e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       1.16751287121657e-02,
       -6.65685876623375e-04,
       -6.65685876623375e-04,
       -6.65685876623375e-04,
       -6.65685876623375e-04,
       6.27082108562500e-06}
    },
    {
      330,
      {{5.55555555555555e-02, 5.55555555555555e-02, 8.33333333333333e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 7.22222222222222e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 8.33333333333333e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 7.22222222222222e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 6.11111111111111e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 7.22222222222222e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {6.11111111111111e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {7.22222222222222e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {8.33333333333333e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 7.22222222222222e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 6.11111111111111e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.00000000000000e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.55555555555555e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       8.32107345778750e-05,
       8.32107345778750e-05,
       8.32107345778750e-05,
       8.32107345778750e-05,
       -3.25756939500000e-07}
    },
    {
      330,
      {{5.55555555555555e-02, 5.55555555555555e-02, 8.33333333333333e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 7.22222222222222e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 8.33333333333333e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 7.22222222222222e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 6.11111111111111e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 7.22222222222222e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {6.11111111111111e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {7.22222222222222e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {8.33333333333333e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 7.22222222222222e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 6.11111111111111e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.00000000000000e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.55555555555555e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       6.43171241036570e-02,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       -1.97838716297478e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       2.26908626208874e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       -1.19852718709862e-01,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       2.91716145775597e-02,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       -2.87387783684075e-03,
       8.32107345778750e-05,
       8.32107345778750e-05,
       8.32107345778750e-05,
       8.32107345778750e-05,
       -3.25756939500000e-07}
    },
    {
      495,
      {{5.00000000000000e-02, 5.00000000000000e-02, 8.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 7.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 8.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 7.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 6.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 7.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {6.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {7.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {8.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 7.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 6.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 4.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 8.33333333333333e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 7.22222222222222e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 8.33333333333333e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 7.22222222222222e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 6.11111111111111e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 7.22222222222222e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {6.11111111111111e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {7.22222222222222e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {8.33333333333333e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 7.22222222222222e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 6.11111111111111e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.00000000000000e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.55555555555555e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       -8.22963309012500e-06,
       -8.22963309012500e-06,
       -8.22963309012500e-06,
       -8.22963309012500e-06,
       1.35732058750000e-08}
    },
    {
      495,
      {{5.00000000000000e-02, 5.00000000000000e-02, 8.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 7.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 8.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 7.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 6.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 7.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {6.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {7.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {8.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 7.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 6.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 4.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 8.33333333333333e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 7.22222222222222e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 8.33333333333333e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 7.22222222222222e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 6.11111111111111e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 7.22222222222222e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {6.11111111111111e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {7.22222222222222e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {8.33333333333333e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 7.22222222222222e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 6.11111111111111e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.00000000000000e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.55555555555555e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       8.22063524662432e-02,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       -2.74194055389274e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       3.51713273417738e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       -2.18010248710487e-01,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       6.74171542742971e-02,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       -9.72387152585325e-03,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       5.47405302255375e-04,
       -8.22963309012500e-06,
       -8.22963309012500e-06,
       -8.22963309012500e-06,
       -8.22963309012500e-06,
       1.35732058750000e-08}
    },
    {
      715,
      {{4.54545454545455e-02, 4.54545454545455e-02, 8.63636363636364e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 7.72727272727272e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 6.81818181818182e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 7.72727272727272e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 8.63636363636364e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 7.72727272727272e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 6.81818181818182e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 5.90909090909091e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 6.81818181818182e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 7.72727272727272e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 6.81818181818182e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 5.90909090909091e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 5.00000000000000e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 5.90909090909091e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 6.81818181818182e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {5.00000000000000e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {5.90909090909091e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {5.90909090909091e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {6.81818181818182e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {6.81818181818182e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {7.72727272727272e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {7.72727272727272e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {8.63636363636364e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 7.72727272727272e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 6.81818181818182e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 7.72727272727272e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 6.81818181818182e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 5.90909090909091e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 6.81818181818182e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {5.90909090909091e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {6.81818181818182e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {7.72727272727272e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 6.81818181818182e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 5.90909090909091e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 5.00000000000000e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 4.09090909090909e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 3.18181818181818e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 2.27272727272728e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 1.36363636363637e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 4.54545454545455e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 8.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 7.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 8.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 7.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 6.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 7.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {6.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {7.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {8.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 7.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 6.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 4.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 8.33333333333333e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 7.22222222222222e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 8.33333333333333e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 7.22222222222222e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 6.11111111111111e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 7.22222222222222e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {6.11111111111111e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {7.22222222222222e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {8.33333333333333e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 7.22222222222222e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 6.11111111111111e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.00000000000000e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.55555555555555e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       6.61309801875000e-07,
       6.61309801875000e-07,
       6.61309801875000e-07,
       6.61309801875000e-07,
       -4.64041250000000e-10}
    },
    {
      715,
      {{4.54545454545455e-02, 4.54545454545455e-02, 8.63636363636364e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 7.72727272727272e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 6.81818181818182e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 7.72727272727272e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 8.63636363636364e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 7.72727272727272e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 6.81818181818182e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 5.90909090909091e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 6.81818181818182e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 7.72727272727272e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 6.81818181818182e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 5.90909090909091e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 5.00000000000000e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 5.90909090909091e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 6.81818181818182e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {5.00000000000000e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {5.90909090909091e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {5.90909090909091e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {6.81818181818182e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {6.81818181818182e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {7.72727272727272e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {7.72727272727272e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {8.63636363636364e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 7.72727272727272e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 6.81818181818182e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 7.72727272727272e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 6.81818181818182e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 5.90909090909091e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 6.81818181818182e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {5.90909090909091e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {6.81818181818182e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {7.72727272727272e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 6.81818181818182e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 5.90909090909091e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 5.00000000000000e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 4.09090909090909e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 3.18181818181818e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 2.27272727272728e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 1.36363636363637e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 4.54545454545455e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 8.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 7.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 8.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 7.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 6.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 7.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {6.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {7.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {8.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 7.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 6.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 4.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 8.33333333333333e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 7.22222222222222e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 8.33333333333333e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 7.22222222222222e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 6.11111111111111e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 7.22222222222222e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {6.11111111111111e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {7.22222222222222e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {8.33333333333333e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 7.22222222222222e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 6.11111111111111e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.00000000000000e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.55555555555555e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       1.08823933894312e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       -3.91458821267825e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       5.55242962163281e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       -3.94906131556758e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       1.48368085927970e-01,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       -2.85531476926435e-02,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       2.53225820985763e-03,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       -8.34141412960000e-05,
       6.61309801875000e-07,
       6.61309801875000e-07,
       6.61309801875000e-07,
       6.61309801875000e-07,
       -4.64041250000000e-10}
    },
    {
      1001,
      {{4.16666666666665e-02, 4.16666666666665e-02, 8.75000000000000e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 7.91666666666667e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 7.08333333333333e-01},
       {4.16666666666665e-02, 2.91666666666667e-01, 6.25000000000000e-01},
       {4.16666666666665e-02, 3.75000000000000e-01, 5.41666666666667e-01},
       {4.16666666666665e-02, 4.58333333333333e-01, 4.58333333333333e-01},
       {4.16666666666665e-02, 5.41666666666667e-01, 3.75000000000000e-01},
       {4.16666666666665e-02, 6.25000000000000e-01, 2.91666666666667e-01},
       {4.16666666666665e-02, 7.08333333333333e-01, 2.08333333333333e-01},
       {4.16666666666665e-02, 7.91666666666667e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 8.75000000000000e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 7.91666666666667e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 7.08333333333333e-01},
       {1.25000000000000e-01, 2.08333333333333e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 2.91666666666667e-01, 5.41666666666667e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 4.58333333333333e-01},
       {1.25000000000000e-01, 4.58333333333333e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 5.41666666666667e-01, 2.91666666666667e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 2.08333333333333e-01},
       {1.25000000000000e-01, 7.08333333333333e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 7.91666666666667e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 7.08333333333333e-01},
       {2.08333333333333e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {2.08333333333333e-01, 2.08333333333333e-01, 5.41666666666667e-01},
       {2.08333333333333e-01, 2.91666666666667e-01, 4.58333333333333e-01},
       {2.08333333333333e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {2.08333333333333e-01, 4.58333333333333e-01, 2.91666666666667e-01},
       {2.08333333333333e-01, 5.41666666666667e-01, 2.08333333333333e-01},
       {2.08333333333333e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {2.08333333333333e-01, 7.08333333333333e-01, 4.16666666666665e-02},
       {2.91666666666667e-01, 4.16666666666665e-02, 6.25000000000000e-01},
       {2.91666666666667e-01, 1.25000000000000e-01, 5.41666666666667e-01},
       {2.91666666666667e-01, 2.08333333333333e-01, 4.58333333333333e-01},
       {2.91666666666667e-01, 2.91666666666667e-01, 3.75000000000000e-01},
       {2.91666666666667e-01, 3.75000000000000e-01, 2.91666666666667e-01},
       {2.91666666666667e-01, 4.58333333333333e-01, 2.08333333333333e-01},
       {2.91666666666667e-01, 5.41666666666667e-01, 1.25000000000000e-01},
       {2.91666666666667e-01, 6.25000000000000e-01, 4.16666666666665e-02},
       {3.75000000000000e-01, 4.16666666666665e-02, 5.41666666666667e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 4.58333333333333e-01},
       {3.75000000000000e-01, 2.08333333333333e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 2.91666666666667e-01, 2.91666666666667e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 2.08333333333333e-01},
       {3.75000000000000e-01, 4.58333333333333e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 5.41666666666667e-01, 4.16666666666665e-02},
       {4.58333333333333e-01, 4.16666666666665e-02, 4.58333333333333e-01},
       {4.58333333333333e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {4.58333333333333e-01, 2.08333333333333e-01, 2.91666666666667e-01},
       {4.58333333333333e-01, 2.91666666666667e-01, 2.08333333333333e-01},
       {4.58333333333333e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {4.58333333333333e-01, 4.58333333333333e-01, 4.16666666666665e-02},
       {5.41666666666667e-01, 4.16666666666665e-02, 3.75000000000000e-01},
       {5.41666666666667e-01, 1.25000000000000e-01, 2.91666666666667e-01},
       {5.41666666666667e-01, 2.08333333333333e-01, 2.08333333333333e-01},
       {5.41666666666667e-01, 2.91666666666667e-01, 1.25000000000000e-01},
       {5.41666666666667e-01, 3.75000000000000e-01, 4.16666666666665e-02},
       {6.25000000000000e-01, 4.16666666666665e-02, 2.91666666666667e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 2.08333333333333e-01},
       {6.25000000000000e-01, 2.08333333333333e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 2.91666666666667e-01, 4.16666666666665e-02},
       {7.08333333333333e-01, 4.16666666666665e-02, 2.08333333333333e-01},
       {7.08333333333333e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {7.08333333333333e-01, 2.08333333333333e-01, 4.16666666666665e-02},
       {7.91666666666667e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {7.91666666666667e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {8.75000000000000e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 7.91666666666667e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 7.08333333333333e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 6.25000000000000e-01},
       {4.16666666666665e-02, 2.91666666666667e-01, 5.41666666666667e-01},
       {4.16666666666665e-02, 3.75000000000000e-01, 4.58333333333333e-01},
       {4.16666666666665e-02, 4.58333333333333e-01, 3.75000000000000e-01},
       {4.16666666666665e-02, 5.41666666666667e-01, 2.91666666666667e-01},
       {4.16666666666665e-02, 6.25000000000000e-01, 2.08333333333333e-01},
       {4.16666666666665e-02, 7.08333333333333e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 7.91666666666667e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 7.08333333333333e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 2.08333333333333e-01, 5.41666666666667e-01},
       {1.25000000000000e-01, 2.91666666666667e-01, 4.58333333333333e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 4.58333333333333e-01, 2.91666666666667e-01},
       {1.25000000000000e-01, 5.41666666666667e-01, 2.08333333333333e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 7.08333333333333e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 6.25000000000000e-01},
       {2.08333333333333e-01, 1.25000000000000e-01, 5.41666666666667e-01},
       {2.08333333333333e-01, 2.08333333333333e-01, 4.58333333333333e-01},
       {2.08333333333333e-01, 2.91666666666667e-01, 3.75000000000000e-01},
       {2.08333333333333e-01, 3.75000000000000e-01, 2.91666666666667e-01},
       {2.08333333333333e-01, 4.58333333333333e-01, 2.08333333333333e-01},
       {2.08333333333333e-01, 5.41666666666667e-01, 1.25000000000000e-01},
       {2.08333333333333e-01, 6.25000000000000e-01, 4.16666666666665e-02},
       {2.91666666666667e-01, 4.16666666666665e-02, 5.41666666666667e-01},
       {2.91666666666667e-01, 1.25000000000000e-01, 4.58333333333333e-01},
       {2.91666666666667e-01, 2.08333333333333e-01, 3.75000000000000e-01},
       {2.91666666666667e-01, 2.91666666666667e-01, 2.91666666666667e-01},
       {2.91666666666667e-01, 3.75000000000000e-01, 2.08333333333333e-01},
       {2.91666666666667e-01, 4.58333333333333e-01, 1.25000000000000e-01},
       {2.91666666666667e-01, 5.41666666666667e-01, 4.16666666666665e-02},
       {3.75000000000000e-01, 4.16666666666665e-02, 4.58333333333333e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 2.08333333333333e-01, 2.91666666666667e-01},
       {3.75000000000000e-01, 2.91666666666667e-01, 2.08333333333333e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 4.58333333333333e-01, 4.16666666666665e-02},
       {4.58333333333333e-01, 4.16666666666665e-02, 3.75000000000000e-01},
       {4.58333333333333e-01, 1.25000000000000e-01, 2.91666666666667e-01},
       {4.58333333333333e-01, 2.08333333333333e-01, 2.08333333333333e-01},
       {4.58333333333333e-01, 2.91666666666667e-01, 1.25000000000000e-01},
       {4.58333333333333e-01, 3.75000000000000e-01, 4.16666666666665e-02},
       {5.41666666666667e-01, 4.16666666666665e-02, 2.91666666666667e-01},
       {5.41666666666667e-01, 1.25000000000000e-01, 2.08333333333333e-01},
       {5.41666666666667e-01, 2.08333333333333e-01, 1.25000000000000e-01},
       {5.41666666666667e-01, 2.91666666666667e-01, 4.16666666666665e-02},
       {6.25000000000000e-01, 4.16666666666665e-02, 2.08333333333333e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 2.08333333333333e-01, 4.16666666666665e-02},
       {7.08333333333333e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {7.08333333333333e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {7.91666666666667e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 7.08333333333333e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 6.25000000000000e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 5.41666666666667e-01},
       {4.16666666666665e-02, 2.91666666666667e-01, 4.58333333333333e-01},
       {4.16666666666665e-02, 3.75000000000000e-01, 3.75000000000000e-01},
       {4.16666666666665e-02, 4.58333333333333e-01, 2.91666666666667e-01},
       {4.16666666666665e-02, 5.41666666666667e-01, 2.08333333333333e-01},
       {4.16666666666665e-02, 6.25000000000000e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 7.08333333333333e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 6.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 5.41666666666667e-01},
       {1.25000000000000e-01, 2.08333333333333e-01, 4.58333333333333e-01},
       {1.25000000000000e-01, 2.91666666666667e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 2.91666666666667e-01},
       {1.25000000000000e-01, 4.58333333333333e-01, 2.08333333333333e-01},
       {1.25000000000000e-01, 5.41666666666667e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 5.41666666666667e-01},
       {2.08333333333333e-01, 1.25000000000000e-01, 4.58333333333333e-01},
       {2.08333333333333e-01, 2.08333333333333e-01, 3.75000000000000e-01},
       {2.08333333333333e-01, 2.91666666666667e-01, 2.91666666666667e-01},
       {2.08333333333333e-01, 3.75000000000000e-01, 2.08333333333333e-01},
       {2.08333333333333e-01, 4.58333333333333e-01, 1.25000000000000e-01},
       {2.08333333333333e-01, 5.41666666666667e-01, 4.16666666666665e-02},
       {2.91666666666667e-01, 4.16666666666665e-02, 4.58333333333333e-01},
       {2.91666666666667e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {2.91666666666667e-01, 2.08333333333333e-01, 2.91666666666667e-01},
       {2.91666666666667e-01, 2.91666666666667e-01, 2.08333333333333e-01},
       {2.91666666666667e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {2.91666666666667e-01, 4.58333333333333e-01, 4.16666666666665e-02},
       {3.75000000000000e-01, 4.16666666666665e-02, 3.75000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 2.91666666666667e-01},
       {3.75000000000000e-01, 2.08333333333333e-01, 2.08333333333333e-01},
       {3.75000000000000e-01, 2.91666666666667e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 4.16666666666665e-02},
       {4.58333333333333e-01, 4.16666666666665e-02, 2.91666666666667e-01},
       {4.58333333333333e-01, 1.25000000000000e-01, 2.08333333333333e-01},
       {4.58333333333333e-01, 2.08333333333333e-01, 1.25000000000000e-01},
       {4.58333333333333e-01, 2.91666666666667e-01, 4.16666666666665e-02},
       {5.41666666666667e-01, 4.16666666666665e-02, 2.08333333333333e-01},
       {5.41666666666667e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {5.41666666666667e-01, 2.08333333333333e-01, 4.16666666666665e-02},
       {6.25000000000000e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {7.08333333333333e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 6.25000000000000e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 5.41666666666667e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 4.58333333333333e-01},
       {4.16666666666665e-02, 2.91666666666667e-01, 3.75000000000000e-01},
       {4.16666666666665e-02, 3.75000000000000e-01, 2.91666666666667e-01},
       {4.16666666666665e-02, 4.58333333333333e-01, 2.08333333333333e-01},
       {4.16666666666665e-02, 5.41666666666667e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 6.25000000000000e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 5.41666666666667e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 4.58333333333333e-01},
       {1.25000000000000e-01, 2.08333333333333e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 2.91666666666667e-01, 2.91666666666667e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 2.08333333333333e-01},
       {1.25000000000000e-01, 4.58333333333333e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 5.41666666666667e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 4.58333333333333e-01},
       {2.08333333333333e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {2.08333333333333e-01, 2.08333333333333e-01, 2.91666666666667e-01},
       {2.08333333333333e-01, 2.91666666666667e-01, 2.08333333333333e-01},
       {2.08333333333333e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {2.08333333333333e-01, 4.58333333333333e-01, 4.16666666666665e-02},
       {2.91666666666667e-01, 4.16666666666665e-02, 3.75000000000000e-01},
       {2.91666666666667e-01, 1.25000000000000e-01, 2.91666666666667e-01},
       {2.91666666666667e-01, 2.08333333333333e-01, 2.08333333333333e-01},
       {2.91666666666667e-01, 2.91666666666667e-01, 1.25000000000000e-01},
       {2.91666666666667e-01, 3.75000000000000e-01, 4.16666666666665e-02},
       {3.75000000000000e-01, 4.16666666666665e-02, 2.91666666666667e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 2.08333333333333e-01},
       {3.75000000000000e-01, 2.08333333333333e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 2.91666666666667e-01, 4.16666666666665e-02},
       {4.58333333333333e-01, 4.16666666666665e-02, 2.08333333333333e-01},
       {4.58333333333333e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {4.58333333333333e-01, 2.08333333333333e-01, 4.16666666666665e-02},
       {5.41666666666667e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {5.41666666666667e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {6.25000000000000e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 5.41666666666667e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 4.58333333333333e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 3.75000000000000e-01},
       {4.16666666666665e-02, 2.91666666666667e-01, 2.91666666666667e-01},
       {4.16666666666665e-02, 3.75000000000000e-01, 2.08333333333333e-01},
       {4.16666666666665e-02, 4.58333333333333e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 5.41666666666667e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 4.58333333333333e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 2.08333333333333e-01, 2.91666666666667e-01},
       {1.25000000000000e-01, 2.91666666666667e-01, 2.08333333333333e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 4.58333333333333e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 3.75000000000000e-01},
       {2.08333333333333e-01, 1.25000000000000e-01, 2.91666666666667e-01},
       {2.08333333333333e-01, 2.08333333333333e-01, 2.08333333333333e-01},
       {2.08333333333333e-01, 2.91666666666667e-01, 1.25000000000000e-01},
       {2.08333333333333e-01, 3.75000000000000e-01, 4.16666666666665e-02},
       {2.91666666666667e-01, 4.16666666666665e-02, 2.91666666666667e-01},
       {2.91666666666667e-01, 1.25000000000000e-01, 2.08333333333333e-01},
       {2.91666666666667e-01, 2.08333333333333e-01, 1.25000000000000e-01},
       {2.91666666666667e-01, 2.91666666666667e-01, 4.16666666666665e-02},
       {3.75000000000000e-01, 4.16666666666665e-02, 2.08333333333333e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 2.08333333333333e-01, 4.16666666666665e-02},
       {4.58333333333333e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {4.58333333333333e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {5.41666666666667e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 4.58333333333333e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 3.75000000000000e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 2.91666666666667e-01},
       {4.16666666666665e-02, 2.91666666666667e-01, 2.08333333333333e-01},
       {4.16666666666665e-02, 3.75000000000000e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 4.58333333333333e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 3.75000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 2.91666666666667e-01},
       {1.25000000000000e-01, 2.08333333333333e-01, 2.08333333333333e-01},
       {1.25000000000000e-01, 2.91666666666667e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 2.91666666666667e-01},
       {2.08333333333333e-01, 1.25000000000000e-01, 2.08333333333333e-01},
       {2.08333333333333e-01, 2.08333333333333e-01, 1.25000000000000e-01},
       {2.08333333333333e-01, 2.91666666666667e-01, 4.16666666666665e-02},
       {2.91666666666667e-01, 4.16666666666665e-02, 2.08333333333333e-01},
       {2.91666666666667e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {2.91666666666667e-01, 2.08333333333333e-01, 4.16666666666665e-02},
       {3.75000000000000e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {4.58333333333333e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 3.75000000000000e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 2.91666666666667e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 2.08333333333333e-01},
       {4.16666666666665e-02, 2.91666666666667e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 3.75000000000000e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 2.91666666666667e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 2.08333333333333e-01},
       {1.25000000000000e-01, 2.08333333333333e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 2.91666666666667e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 2.08333333333333e-01},
       {2.08333333333333e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {2.08333333333333e-01, 2.08333333333333e-01, 4.16666666666665e-02},
       {2.91666666666667e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {2.91666666666667e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {3.75000000000000e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 2.91666666666667e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 2.08333333333333e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 2.91666666666667e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 2.08333333333333e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 2.08333333333333e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {2.08333333333333e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {2.91666666666667e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 2.08333333333333e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 1.25000000000000e-01},
       {4.16666666666665e-02, 2.08333333333333e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 4.16666666666665e-02},
       {2.08333333333333e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 1.25000000000000e-01},
       {4.16666666666665e-02, 1.25000000000000e-01, 4.16666666666665e-02},
       {1.25000000000000e-01, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.16666666666665e-02, 4.16666666666665e-02, 4.16666666666665e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 8.63636363636364e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 7.72727272727272e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 6.81818181818182e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 7.72727272727272e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 8.63636363636364e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 7.72727272727272e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 6.81818181818182e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 5.90909090909091e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 6.81818181818182e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 7.72727272727272e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 6.81818181818182e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 5.90909090909091e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 5.00000000000000e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 5.90909090909091e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 6.81818181818182e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {5.00000000000000e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {5.90909090909091e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {5.90909090909091e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {6.81818181818182e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {6.81818181818182e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {7.72727272727272e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {7.72727272727272e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {8.63636363636364e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 7.72727272727272e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 6.81818181818182e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 7.72727272727272e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 6.81818181818182e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 5.90909090909091e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 6.81818181818182e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {5.90909090909091e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {6.81818181818182e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {7.72727272727272e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 6.81818181818182e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 5.90909090909091e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 6.81818181818182e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 5.90909090909091e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 5.00000000000000e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 5.90909090909091e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {5.00000000000000e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {5.90909090909091e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {6.81818181818182e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 5.90909090909091e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 5.00000000000000e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 5.90909090909091e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 5.00000000000000e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 4.09090909090909e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 5.00000000000000e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {4.09090909090909e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {5.00000000000000e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {5.90909090909091e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 5.00000000000000e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 4.09090909090909e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 5.00000000000000e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 4.09090909090909e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 3.18181818181818e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 4.09090909090909e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {3.18181818181818e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {4.09090909090909e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {5.00000000000000e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 4.09090909090909e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 3.18181818181818e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 4.09090909090909e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 3.18181818181818e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 2.27272727272728e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 3.18181818181818e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {2.27272727272728e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {3.18181818181818e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {4.09090909090909e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 3.18181818181818e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 2.27272727272728e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 3.18181818181818e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 2.27272727272728e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 1.36363636363637e-01},
       {1.36363636363637e-01, 2.27272727272728e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {2.27272727272728e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {3.18181818181818e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 2.27272727272728e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 1.36363636363637e-01},
       {4.54545454545455e-02, 2.27272727272728e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 1.36363636363637e-01},
       {1.36363636363637e-01, 1.36363636363637e-01, 4.54545454545455e-02},
       {2.27272727272728e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 1.36363636363637e-01},
       {4.54545454545455e-02, 1.36363636363637e-01, 4.54545454545455e-02},
       {1.36363636363637e-01, 4.54545454545455e-02, 4.54545454545455e-02},
       {4.54545454545455e-02, 4.54545454545455e-02, 4.54545454545455e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 8.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 7.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 8.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 7.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 6.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 7.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {6.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {7.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {8.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 7.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 6.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 7.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 6.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 6.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {6.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {7.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 6.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 6.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 4.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 5.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {4.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {6.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 4.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 5.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 4.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 3.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 4.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {3.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {4.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {5.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 4.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 3.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 4.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 3.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 2.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 3.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {3.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {4.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 3.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 2.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 3.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 2.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 1.50000000000000e-01},
       {1.50000000000000e-01, 2.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {2.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {3.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 2.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 1.50000000000000e-01},
       {5.00000000000000e-02, 2.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 1.50000000000000e-01},
       {1.50000000000000e-01, 1.50000000000000e-01, 5.00000000000000e-02},
       {2.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 1.50000000000000e-01},
       {5.00000000000000e-02, 1.50000000000000e-01, 5.00000000000000e-02},
       {1.50000000000000e-01, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.00000000000000e-02, 5.00000000000000e-02, 5.00000000000000e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 8.33333333333333e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 7.22222222222222e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 8.33333333333333e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 7.22222222222222e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 6.11111111111111e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 7.22222222222222e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {6.11111111111111e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {7.22222222222222e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {8.33333333333333e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 7.22222222222222e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 6.11111111111111e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 7.22222222222222e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 6.11111111111111e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 6.11111111111111e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {6.11111111111111e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {7.22222222222222e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 6.11111111111111e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.00000000000000e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 6.11111111111111e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.00000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 3.88888888888889e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {3.88888888888889e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {6.11111111111111e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.00000000000000e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 3.88888888888889e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 5.00000000000000e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 3.88888888888889e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 2.77777777777778e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 3.88888888888889e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.77777777777778e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {3.88888888888889e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {5.00000000000000e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 3.88888888888889e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 2.77777777777778e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 3.88888888888889e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 2.77777777777778e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 2.77777777777778e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {2.77777777777778e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {3.88888888888889e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 2.77777777777778e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 1.66666666666667e-01},
       {5.55555555555555e-02, 2.77777777777778e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.55555555555555e-02},
       {2.77777777777778e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 1.66666666666667e-01},
       {5.55555555555555e-02, 1.66666666666667e-01, 5.55555555555555e-02},
       {1.66666666666667e-01, 5.55555555555555e-02, 5.55555555555555e-02},
       {5.55555555555555e-02, 5.55555555555555e-02, 5.55555555555555e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 8.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 8.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 5.62500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 6.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {5.62500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {8.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 5.62500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 6.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 5.62500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 4.37500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 5.62500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {4.37500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {5.62500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {6.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 5.62500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 4.37500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 5.62500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 4.37500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 3.12500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 4.37500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {3.12500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {4.37500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {5.62500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 4.37500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 3.12500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 4.37500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 3.12500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 1.87500000000000e-01},
       {1.87500000000000e-01, 3.12500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {3.12500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {4.37500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 3.12500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 1.87500000000000e-01},
       {6.25000000000000e-02, 3.12500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 1.87500000000000e-01},
       {1.87500000000000e-01, 1.87500000000000e-01, 6.25000000000000e-02},
       {3.12500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 1.87500000000000e-01},
       {6.25000000000000e-02, 1.87500000000000e-01, 6.25000000000000e-02},
       {1.87500000000000e-01, 6.25000000000000e-02, 6.25000000000000e-02},
       {6.25000000000000e-02, 6.25000000000000e-02, 6.25000000000000e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.85714285714285e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 6.42857142857143e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 7.85714285714285e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 6.42857142857143e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 5.00000000000000e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 6.42857142857143e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {5.00000000000000e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {6.42857142857143e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {7.85714285714285e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 6.42857142857143e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 5.00000000000000e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 6.42857142857143e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 5.00000000000000e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 3.57142857142857e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 5.00000000000000e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {3.57142857142857e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {5.00000000000000e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {6.42857142857143e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 5.00000000000000e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 3.57142857142857e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 5.00000000000000e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 3.57142857142857e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 2.14285714285715e-01},
       {2.14285714285715e-01, 3.57142857142857e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {3.57142857142857e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {5.00000000000000e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 3.57142857142857e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 2.14285714285715e-01},
       {7.14285714285715e-02, 3.57142857142857e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 2.14285714285715e-01},
       {2.14285714285715e-01, 2.14285714285715e-01, 7.14285714285715e-02},
       {3.57142857142857e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 2.14285714285715e-01},
       {7.14285714285715e-02, 2.14285714285715e-01, 7.14285714285715e-02},
       {2.14285714285715e-01, 7.14285714285715e-02, 7.14285714285715e-02},
       {7.14285714285715e-02, 7.14285714285715e-02, 7.14285714285715e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 7.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 5.83333333333333e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 7.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 5.83333333333333e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 4.16666666666667e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 5.83333333333333e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {4.16666666666667e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {5.83333333333333e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {7.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 5.83333333333333e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 4.16666666666667e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 5.83333333333333e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 4.16666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01},
       {2.50000000000000e-01, 4.16666666666667e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {4.16666666666667e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {5.83333333333333e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 4.16666666666667e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 2.50000000000000e-01},
       {8.33333333333335e-02, 4.16666666666667e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 2.50000000000000e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 8.33333333333335e-02},
       {4.16666666666667e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 2.50000000000000e-01},
       {8.33333333333335e-02, 2.50000000000000e-01, 8.33333333333335e-02},
       {2.50000000000000e-01, 8.33333333333335e-02, 8.33333333333335e-02},
       {8.33333333333335e-02, 8.33333333333335e-02, 8.33333333333335e-02},
       {1.00000000000000e-01, 1.00000000000000e-01, 7.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 7.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {5.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {7.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 5.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 5.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {3.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {5.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 3.00000000000000e-01},
       {1.00000000000000e-01, 3.00000000000000e-01, 1.00000000000000e-01},
       {3.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.00000000000000e-01, 1.00000000000000e-01, 1.00000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 6.25000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 6.25000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {3.75000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {6.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 3.75000000000000e-01},
       {1.25000000000000e-01, 3.75000000000000e-01, 1.25000000000000e-01},
       {3.75000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.25000000000000e-01, 1.25000000000000e-01, 1.25000000000000e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 5.00000000000000e-01},
       {1.66666666666667e-01, 5.00000000000000e-01, 1.66666666666667e-01},
       {5.00000000000000e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {1.66666666666667e-01, 1.66666666666667e-01, 1.66666666666667e-01},
       {2.50000000000000e-01, 2.50000000000000e-01, 2.50000000000000e-01}},
      {1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       1.48296360441426e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       -5.72508521791816e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       8.89679139245057e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       -7.13883808495647e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       3.15924905245407e-01,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       -7.65266969523215e-02,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       9.51771589754788e-03,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       -5.31987018877625e-04,
       1.04267676620000e-05,
       1.04267676620000e-05,
       1.04267676620000e-05,
       1.04267676620000e-05,
       1.04267676620000e-05,
       1.04267676620000e-05,
       1.04267676620000e-05,
       1.04267676620000e-05,
       1.04267676620000e-05,
       1.04267676620000e-05,
       -4.40873201250000e-08,
       -4.40873201250000e-08,
       -4.40873201250000e-08,
       -4.40873201250000e-08,
       1.32583750000000e-11}
    } // end Tet

  };
    
} // end namespace Intrepid2
