// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright (C) 2019 Intel Corporation <www.intel.com>
 */

#include <dm.h>
#include <log.h>
#include <ns16550.h>
#include <serial.h>
#include <asm/arch/slimbootloader.h>
#include <asm/global_data.h>

/**
 * The serial port info hob is generated by Slim Bootloader, so eligible for
 * Slim Bootloader based boards only.
 */
static int slimbootloader_serial_of_to_plat(struct udevice *dev)
{
	const efi_guid_t guid = SBL_SERIAL_PORT_INFO_GUID;
	struct sbl_serial_port_info *data;
	struct ns16550_plat *plat = dev_get_plat(dev);

	if (!gd->arch.hob_list)
		panic("hob list not found!");

	data = hob_get_guid_hob_data(gd->arch.hob_list, NULL, &guid);
	if (!data) {
		debug("failed to get serial port information\n");
		return -ENOENT;
	}
	debug("type:%d base=0x%08x baudrate=%d stride=%d clk=%d\n",
	      data->type,
	      data->base,
	      data->baud,
	      data->stride,
	      data->clk);

	plat->base = data->base;
	/* ns16550 uses reg_shift, then covert stride to shift */
	plat->reg_shift = data->stride >> 1;
	plat->reg_width = data->stride;
	plat->clock = data->clk;
	plat->fcr = UART_FCR_DEFVAL;
	plat->flags = 0;
	if (data->type == 1)
		plat->flags |= NS16550_FLAG_IO;

	return 0;
}

static const struct udevice_id slimbootloader_serial_ids[] = {
	{ .compatible = "intel,slimbootloader-uart" },
	{}
};

U_BOOT_DRIVER(serial_slimbootloader) = {
	.name	= "serial_slimbootloader",
	.id	= UCLASS_SERIAL,
	.of_match = slimbootloader_serial_ids,
	.of_to_plat = slimbootloader_serial_of_to_plat,
	.plat_auto	= sizeof(struct ns16550_plat),
	.priv_auto	= sizeof(struct ns16550),
	.probe	= ns16550_serial_probe,
	.ops	= &ns16550_serial_ops,
};
